// Photosynthesis Interactive Application
// This script manages the interactive photosynthesis simulation for Primary 5-6 students

class PhotosynthesisSimulator {
    constructor() {
        // Initialize DOM elements
        this.sunlightSlider = document.getElementById('sunlight');
        this.waterSlider = document.getElementById('water');
        this.co2Slider = document.getElementById('co2');
        
        this.sunlightValue = document.getElementById('sunlight-value');
        this.waterValue = document.getElementById('water-value');
        this.co2Value = document.getElementById('co2-value');
        
        this.photosynthesisRate = document.getElementById('photosynthesis-rate');
        this.oxygenProduction = document.getElementById('oxygen-production');
        this.glucoseFormation = document.getElementById('glucose-formation');
        
        this.resetBtn = document.getElementById('reset-btn');
        this.autoDemoBtn = document.getElementById('auto-demo');
        
        this.analyticsPanel = document.getElementById('analytics-panel');
        this.tooltip = document.getElementById('tooltip');
        
        // Analytics tracking
        this.analytics = {
            interactions: 0,
            startTime: Date.now(),
            conceptsExplored: new Set(),
            sliderChanges: {
                sunlight: 0,
                water: 0,
                co2: 0
            }
        };
        
        // Animation states
        this.isAutoDemo = false;
        this.animationFrameId = null;
        
        // Initialize the application
        this.init();
    }
    
    init() {
        // Set up event listeners for all interactive elements
        this.setupEventListeners();
        
        // Initialize visual state
        this.updateVisualization();
        
        // Start analytics timer
        this.startAnalyticsTimer();
        
        // Show analytics panel after 3 seconds
        setTimeout(() => {
            this.analyticsPanel.classList.add('show');
        }, 3000);
        
        console.log('Photosynthesis Simulator initialized');
    }
    
    setupEventListeners() {
        // Slider event listeners with touch support
        this.sunlightSlider.addEventListener('input', (e) => this.handleSliderChange('sunlight', e));
        this.waterSlider.addEventListener('input', (e) => this.handleSliderChange('water', e));
        this.co2Slider.addEventListener('input', (e) => this.handleSliderChange('co2', e));
        
        // Touch events for mobile support
        [this.sunlightSlider, this.waterSlider, this.co2Slider].forEach(slider => {
            slider.addEventListener('touchstart', this.handleTouchStart.bind(this), { passive: false });
            slider.addEventListener('touchmove', this.handleTouchMove.bind(this), { passive: false });
            slider.addEventListener('touchend', this.handleTouchEnd.bind(this), { passive: false });
        });
        
        // Button event listeners
        this.resetBtn.addEventListener('click', () => this.resetSimulation());
        this.autoDemoBtn.addEventListener('click', () => this.toggleAutoDemo());
        
        // SVG element interactions for educational tooltips
        this.setupSVGInteractions();
        
        // Keyboard accessibility
        document.addEventListener('keydown', (e) => this.handleKeyboard(e));
    }
    
    handleSliderChange(type, event) {
        const value = parseInt(event.target.value);
        
        // Update display value
        document.getElementById(`${type}-value`).textContent = `${value}%`;
        
        // Track analytics
        this.analytics.interactions++;
        this.analytics.sliderChanges[type]++;
        this.analytics.conceptsExplored.add(type);
        
        // Update visualization
        this.updateVisualization();
        
        // Update analytics display
        this.updateAnalyticsDisplay();
        
        // Provide haptic feedback on mobile
        if ('vibrate' in navigator) {
            navigator.vibrate(10);
        }
        
        console.log(`${type} changed to ${value}%`);
    }
    
    handleTouchStart(event) {
        // Prevent default to handle touch events properly
        event.preventDefault();
        
        // Add visual feedback
        event.target.classList.add('active');
    }
    
    handleTouchMove(event) {
        event.preventDefault();
        
        // Calculate slider value based on touch position
        const slider = event.target;
        const rect = slider.getBoundingClientRect();
        const percentage = Math.max(0, Math.min(100, 
            ((event.touches[0].clientX - rect.left) / rect.width) * 100
        ));
        
        slider.value = Math.round(percentage);
        
        // Trigger input event
        slider.dispatchEvent(new Event('input'));
    }
    
    handleTouchEnd(event) {
        event.preventDefault();
        event.target.classList.remove('active');
    }
    
    updateVisualization() {
        const sunlight = parseInt(this.sunlightSlider.value);
        const water = parseInt(this.waterSlider.value);
        const co2 = parseInt(this.co2Slider.value);
        
        // Calculate photosynthesis rate based on limiting factor principle
        const photosynthesisRate = Math.min(sunlight, water, co2);
        
        // Update plant visual elements based on conditions
        this.updatePlantAppearance(sunlight, water, co2, photosynthesisRate);
        
        // Update process indicators
        this.updateProcessIndicators(photosynthesisRate);
        
        // Update results panel
        this.updateResults(photosynthesisRate);
        
        // Update chemical equation highlighting
        this.updateEquationHighlighting(sunlight, water, co2);
    }
    
    updatePlantAppearance(sunlight, water, co2, rate) {
        // Update leaf color based on photosynthesis rate
        const leaves = document.querySelectorAll('.leaf');
        const leafColor = this.getLeafColor(rate);
        
        leaves.forEach(leaf => {
            leaf.style.fill = leafColor;
            leaf.style.opacity = Math.max(0.6, rate / 100);
        });
        
        // Update sun brightness
        const sun = document.querySelector('.sun');
        sun.style.opacity = sunlight / 100;
        sun.style.filter = `drop-shadow(0 0 ${sunlight / 5}px rgba(255, 215, 0, 0.8))`;
        
        // Update water flow animation speed
        const waterParticles = document.querySelectorAll('.water-particle');
        waterParticles.forEach(particle => {
            const animationDuration = Math.max(0.5, 3 - (water / 50));
            particle.style.animationDuration = `${animationDuration}s`;
        });
        
        // Update CO2 particle movement
        const co2Particles = document.querySelectorAll('.co2-particle');
        co2Particles.forEach(particle => {
            const animationDuration = Math.max(0.8, 4 - (co2 / 25));
            particle.style.animationDuration = `${animationDuration}s`;
        });
        
        // Update oxygen production
        const oxygenParticles = document.querySelectorAll('.oxygen-particle');
        oxygenParticles.forEach(particle => {
            particle.style.opacity = rate / 100;
            const animationDuration = Math.max(0.5, 2 - (rate / 100));
            particle.style.animationDuration = `${animationDuration}s`;
        });
    }
    
    getLeafColor(rate) {
        // Calculate leaf color based on photosynthesis rate
        if (rate < 20) return '#8B4513'; // Brown (dying)
        if (rate < 40) return '#9ACD32'; // Yellow-green (stressed)
        if (rate < 60) return '#228B22'; // Forest green (healthy)
        if (rate < 80) return '#32CD32'; // Lime green (very healthy)
        return '#00FF32'; // Bright green (optimal)
    }
    
    updateProcessIndicators(rate) {
        // Update glucose particles visibility
        const glucoseParticles = document.querySelectorAll('.glucose-particle');
        glucoseParticles.forEach(particle => {
            particle.style.opacity = rate / 100;
            if (rate > 50) {
                particle.classList.add('pulse');
            } else {
                particle.classList.remove('pulse');
            }
        });
    }
    
    updateResults(rate) {
        // Update photosynthesis rate display
        this.photosynthesisRate.textContent = `${Math.round(rate)}%`;
        this.photosynthesisRate.style.color = this.getRateColor(rate);
        
        // Update oxygen production
        const oxygenLevel = this.getProductionLevel(rate);
        this.oxygenProduction.textContent = oxygenLevel;
        this.oxygenProduction.style.color = this.getRateColor(rate);
        
        // Update glucose formation
        const glucoseLevel = this.getProductionLevel(rate);
        this.glucoseFormation.textContent = glucoseLevel;
        this.glucoseFormation.style.color = this.getRateColor(rate);
    }
    
    getProductionLevel(rate) {
        if (rate < 20) return 'Very Low';
        if (rate < 40) return 'Low';
        if (rate < 60) return 'Medium';
        if (rate < 80) return 'High';
        return 'Very High';
    }
    
    getRateColor(rate) {
        if (rate < 30) return '#F44336'; // Red
        if (rate < 60) return '#FF9800'; // Orange
        return '#4CAF50'; // Green
    }
    
    updateEquationHighlighting(sunlight, water, co2) {
        const equation = document.querySelector('.equation');
        
        // Highlight limiting factor
        const minValue = Math.min(sunlight, water, co2);
        let highlightText = '';
        
        if (sunlight === minValue && sunlight < 70) {
            highlightText = 'Light Energy is limiting photosynthesis';
        } else if (water === minValue && water < 70) {
            highlightText = 'Water is limiting photosynthesis';
        } else if (co2 === minValue && co2 < 70) {
            highlightText = 'Carbon Dioxide is limiting photosynthesis';
        } else {
            highlightText = 'Optimal conditions for photosynthesis';
        }
        
        // Update equation styling
        const energySpan = equation.querySelector('.highlight-energy');
        if (sunlight < 30) {
            energySpan.style.color = '#F44336';
            energySpan.style.textShadow = '0 0 4px rgba(244, 67, 54, 0.5)';
        } else {
            energySpan.style.color = '#FF9800';
            energySpan.style.textShadow = '0 0 4px rgba(255, 152, 0, 0.5)';
        }
    }
    
    setupSVGInteractions() {
        // Add interactive tooltips to SVG elements
        const interactiveElements = [
            { selector: '.leaf', text: 'Leaves: Where photosynthesis happens! Chloroplasts capture sunlight.' },
            { selector: '.sun', text: 'Sun: Provides light energy needed for photosynthesis.' },
            { selector: '.co2-particle', text: 'Carbon Dioxide: Plants absorb CO₂ from the air through stomata.' },
            { selector: '.water-particle', text: 'Water: Absorbed by roots and transported to leaves.' },
            { selector: '.oxygen-particle', text: 'Oxygen: Released as a byproduct of photosynthesis.' },
            { selector: '.glucose-particle', text: 'Glucose: Sugar produced by plants for energy and growth.' },
            { selector: '.root-system', text: 'Roots: Absorb water and minerals from the soil.' }
        ];
        
        interactiveElements.forEach(element => {
            const elements = document.querySelectorAll(element.selector);
            elements.forEach(el => {
                el.addEventListener('mouseenter', (e) => this.showTooltip(e, element.text));
                el.addEventListener('mouseleave', () => this.hideTooltip());
                el.addEventListener('click', (e) => this.showTooltip(e, element.text));
                
                // Touch support
                el.addEventListener('touchstart', (e) => {
                    e.preventDefault();
                    this.showTooltip(e, element.text);
                    setTimeout(() => this.hideTooltip(), 3000);
                });
            });
        });
    }
    
    showTooltip(event, text) {
        const rect = event.target.getBoundingClientRect();
        const containerRect = document.querySelector('.container').getBoundingClientRect();
        
        this.tooltip.textContent = text;
        this.tooltip.style.left = `${rect.left - containerRect.left + rect.width / 2}px`;
        this.tooltip.style.top = `${rect.top - containerRect.top - 10}px`;
        this.tooltip.style.transform = 'translateX(-50%) translateY(-100%)';
        this.tooltip.classList.add('show');
        
        // Track concept exploration
        this.analytics.conceptsExplored.add('tooltip');
        this.updateAnalyticsDisplay();
    }
    
    hideTooltip() {
        this.tooltip.classList.remove('show');
    }
    
    resetSimulation() {
        // Reset all sliders to default values
        this.sunlightSlider.value = 50;
        this.waterSlider.value = 50;
        this.co2Slider.value = 50;
        
        // Update displays
        this.sunlightValue.textContent = '50%';
        this.waterValue.textContent = '50%';
        this.co2Value.textContent = '50%';
        
        // Update visualization
        this.updateVisualization();
        
        // Track analytics
        this.analytics.interactions++;
        this.updateAnalyticsDisplay();
        
        // Stop auto demo if running
        if (this.isAutoDemo) {
            this.toggleAutoDemo();
        }
        
        console.log('Simulation reset');
    }
    
    toggleAutoDemo() {
        if (this.isAutoDemo) {
            // Stop auto demo
            this.isAutoDemo = false;
            this.autoDemoBtn.textContent = 'Auto Demo';
            this.autoDemoBtn.style.background = 'linear-gradient(135deg, #4CAF50, #45a049)';
            
            if (this.animationFrameId) {
                cancelAnimationFrame(this.animationFrameId);
            }
        } else {
            // Start auto demo
            this.isAutoDemo = true;
            this.autoDemoBtn.textContent = 'Stop Demo';
            this.autoDemoBtn.style.background = 'linear-gradient(135deg, #F44336, #d32f2f)';
            
            this.runAutoDemo();
        }
        
        // Track analytics
        this.analytics.interactions++;
        this.analytics.conceptsExplored.add('autoDemo');
        this.updateAnalyticsDisplay();
    }
    
    runAutoDemo() {
        if (!this.isAutoDemo) return;
        
        const time = Date.now() / 1000;
        
        // Create sinusoidal variations for realistic demonstration
        const sunlight = Math.round(50 + 40 * Math.sin(time * 0.5));
        const water = Math.round(50 + 30 * Math.sin(time * 0.3 + 1));
        const co2 = Math.round(50 + 35 * Math.sin(time * 0.4 + 2));
        
        // Update sliders
        this.sunlightSlider.value = sunlight;
        this.waterSlider.value = water;
        this.co2Slider.value = co2;
        
        // Update displays
        this.sunlightValue.textContent = `${sunlight}%`;
        this.waterValue.textContent = `${water}%`;
        this.co2Value.textContent = `${co2}%`;
        
        // Update visualization
        this.updateVisualization();
        
        // Continue animation
        this.animationFrameId = requestAnimationFrame(() => this.runAutoDemo());
    }
    
    handleKeyboard(event) {
        // Keyboard accessibility for sliders
        const activeElement = document.activeElement;
        
        if (activeElement.classList.contains('slider')) {
            let currentValue = parseInt(activeElement.value);
            let newValue = currentValue;
            
            switch (event.key) {
                case 'ArrowUp':
                case 'ArrowRight':
                    newValue = Math.min(100, currentValue + 5);
                    break;
                case 'ArrowDown':
                case 'ArrowLeft':
                    newValue = Math.max(0, currentValue - 5);
                    break;
                case 'Home':
                    newValue = 0;
                    break;
                case 'End':
                    newValue = 100;
                    break;
                default:
                    return;
            }
            
            event.preventDefault();
            activeElement.value = newValue;
            activeElement.dispatchEvent(new Event('input'));
        }
        
        // Global keyboard shortcuts
        if (event.ctrlKey || event.metaKey) {
            switch (event.key) {
                case 'r':
                    event.preventDefault();
                    this.resetSimulation();
                    break;
                case 'd':
                    event.preventDefault();
                    this.toggleAutoDemo();
                    break;
            }
        }
    }
    
    startAnalyticsTimer() {
        setInterval(() => {
            const timeSpent = Math.round((Date.now() - this.analytics.startTime) / 1000);
            document.getElementById('time-spent').textContent = `${timeSpent}s`;
        }, 1000);
    }
    
    updateAnalyticsDisplay() {
        document.getElementById('interaction-count').textContent = this.analytics.interactions;
        document.getElementById('concepts-explored').textContent = 
            `${this.analytics.conceptsExplored.size}/7`;
        
        // Show achievement feedback
        if (this.analytics.conceptsExplored.size >= 5 && !this.achievementShown) {
            this.showAchievement('Great Explorer!', 'You\'ve discovered most photosynthesis concepts!');
            this.achievementShown = true;
        }
    }
    
    showAchievement(title, message) {
        // Create temporary achievement notification
        const achievement = document.createElement('div');
        achievement.className = 'achievement-notification';
        achievement.innerHTML = `
            <div class="achievement-title">${title}</div>
            <div class="achievement-message">${message}</div>
        `;
        
        // Add styles
        achievement.style.cssText = `
            position: fixed;
            top: 20px;
            left: 50%;
            transform: translateX(-50%);
            background: linear-gradient(135deg, #4CAF50, #45a049);
            color: white;
            padding: 15px 20px;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
            z-index: 1002;
            animation: slideDown 0.5s ease-out;
        `;
        
        document.body.appendChild(achievement);
        
        // Remove after 3 seconds
        setTimeout(() => {
            achievement.style.animation = 'slideUp 0.5s ease-in forwards';
            setTimeout(() => document.body.removeChild(achievement), 500);
        }, 3000);
    }
}

// Initialize the application when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    const simulator = new PhotosynthesisSimulator();
    
    // Add CSS animations for achievements
    const style = document.createElement('style');
    style.textContent = `
        @keyframes slideDown {
            from { transform: translateX(-50%) translateY(-100%); opacity: 0; }
            to { transform: translateX(-50%) translateY(0); opacity: 1; }
        }
        
        @keyframes slideUp {
            from { transform: translateX(-50%) translateY(0); opacity: 1; }
            to { transform: translateX(-50%) translateY(-100%); opacity: 0; }
        }
        
        .achievement-title {
            font-weight: bold;
            font-size: 14px;
            margin-bottom: 4px;
        }
        
        .achievement-message {
            font-size: 12px;
            opacity: 0.9;
        }
    `;
    document.head.appendChild(style);
    
    console.log('Photosynthesis Interactive loaded successfully');
});